/*
 * Color table functions
 * $Id: ColorTable.cpp 417 2008-01-23 22:30:38Z fred $
 *
 * Copyright (c) 2006 Fred-Markus Stober <mail@fredstober.de>
 *
 * Permission is hereby granted, free of charge, to any person
 * obtaining a copy of this software and associated documentation
 * files (the "Software"), to deal in the Software without
 * restriction, including without limitation the rights to use,
 * copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following
 * conditions:

 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.

 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
 * OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
 * HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
 * WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
 * OTHER DEALINGS IN THE SOFTWARE.
 *
 */

#define WIN32_LEAN_AND_MEAN
#define _USE_MATH_DEFINES
#include <windows.h>
#include <stdlib.h>
#include <math.h>
#include "Message.h"

#define _C(r,g,b) (RGB(r,g,b))

unsigned long cmStdColorMap[] =
{
	_C(0x20,0x1F,0x21), _C(0x26,0x2C,0x2E), _C(0x35,0x26,0x26), _C(0x37,0x2B,0x27),
	_C(0x30,0x2C,0x2E), _C(0x39,0x2B,0x2D), _C(0x32,0x32,0x29), _C(0x3F,0x32,0x29),
	_C(0x38,0x32,0x2E), _C(0x2E,0x33,0x3D), _C(0x33,0x3A,0x3D), _C(0x47,0x33,0x29),
	_C(0x40,0x39,0x2C), _C(0x40,0x39,0x2E), _C(0x47,0x40,0x2C), _C(0x47,0x40,0x2E),
	_C(0x4E,0x40,0x2C), _C(0x4F,0x40,0x2E), _C(0x4E,0x47,0x38), _C(0x58,0x40,0x37),
	_C(0x65,0x47,0x2D), _C(0x6D,0x5D,0x3D), _C(0x74,0x55,0x30), _C(0x75,0x55,0x32),
	_C(0x74,0x5D,0x32), _C(0x74,0x64,0x33), _C(0x7C,0x6C,0x36), _C(0x52,0x31,0x52),
	_C(0x44,0x48,0x42), _C(0x4C,0x56,0x47), _C(0x65,0x5D,0x45), _C(0x6D,0x5D,0x44),
	_C(0x6C,0x5D,0x4E), _C(0x74,0x6C,0x43), _C(0x7C,0x6C,0x42), _C(0x7C,0x6C,0x4B),
	_C(0x6B,0x73,0x4B), _C(0x73,0x73,0x4B), _C(0x7B,0x7B,0x4A), _C(0x6B,0x6C,0x55),
	_C(0x69,0x6D,0x5E), _C(0x7B,0x6C,0x5D), _C(0x6B,0x73,0x53), _C(0x6A,0x74,0x5D),
	_C(0x72,0x7B,0x52), _C(0x7B,0x7B,0x52), _C(0x57,0x74,0x6E), _C(0x68,0x74,0x66),
	_C(0x9C,0x54,0x2B), _C(0x9D,0x54,0x32), _C(0x9D,0x5B,0x35), _C(0x93,0x6B,0x36),
	_C(0xAA,0x73,0x30), _C(0xC4,0x5A,0x27), _C(0xD9,0x52,0x23), _C(0xD8,0x5A,0x20),
	_C(0xDB,0x5A,0x23), _C(0xE5,0x70,0x37), _C(0x83,0x6C,0x4B), _C(0x8C,0x6B,0x4B),
	_C(0x82,0x73,0x5C), _C(0x93,0x73,0x52), _C(0x81,0x7B,0x63), _C(0x81,0x7B,0x6D),
	_C(0x92,0x7B,0x63), _C(0xD9,0x89,0x3B), _C(0xE4,0x98,0x32), _C(0xDF,0xA1,0x33),
	_C(0xE5,0xA0,0x37), _C(0xF0,0xAB,0x3B), _C(0x8A,0x8A,0x59), _C(0xB2,0x9A,0x58),
	_C(0x89,0x82,0x6B), _C(0x9A,0x82,0x62), _C(0x88,0x8B,0x7C), _C(0x90,0x9A,0x7A),
	_C(0xA2,0x82,0x62), _C(0xA1,0x8A,0x69), _C(0xA9,0x99,0x68), _C(0x99,0xA1,0x60),
	_C(0x99,0xA1,0x68), _C(0xCA,0x81,0x48), _C(0xEB,0x8D,0x43), _C(0xC2,0x91,0x60),
	_C(0xC2,0x91,0x68), _C(0xD1,0xA9,0x77), _C(0xC9,0xB9,0x7F), _C(0xF0,0xE2,0x7B),
	_C(0x9F,0x92,0x8B), _C(0xC0,0xB9,0x99), _C(0xE6,0xB8,0x8F), _C(0xC8,0xC1,0x87),
	_C(0xE0,0xC8,0x86), _C(0xF2,0xCC,0x85), _C(0xF5,0xDA,0x83), _C(0xEC,0xDE,0x9D),
	_C(0xF5,0xD2,0x94), _C(0xF5,0xDA,0x94), _C(0xF4,0xE7,0x84), _C(0xF4,0xE1,0x8A),
	_C(0xF4,0xE1,0x93), _C(0xE7,0xD8,0xA7), _C(0xF1,0xD4,0xA5), _C(0xF1,0xDC,0xA5),
	_C(0xF4,0xDB,0xAD), _C(0xF1,0xDC,0xAE), _C(0xF4,0xDB,0xB5), _C(0xF5,0xDB,0xBD),
	_C(0xF4,0xE2,0xAD), _C(0xF5,0xE9,0xAD), _C(0xF4,0xE3,0xBE), _C(0xF5,0xEA,0xBE),
	_C(0xF7,0xF0,0xB6), _C(0xD9,0xD1,0xC1), _C(0xE0,0xD0,0xC0), _C(0xE7,0xD8,0xC0),
	_C(0xF1,0xDD,0xC6), _C(0xE8,0xE1,0xC0), _C(0xF3,0xED,0xC7), _C(0xF6,0xEC,0xCE),
	_C(0xF8,0xF2,0xC7), _C(0xEF,0xEF,0xD0)
};

struct cABC
{
	BYTE A, B, C;
};

BYTE Max(struct cABC abc)
{
	return (max(max(abc.A, abc.B), abc.C));
}

BYTE Min(struct cABC abc)
{
	return (min(min(abc.A, abc.B), abc.C));
}

struct cABC RGB2RGB(unsigned long Color)
{
	struct cABC rgb;
	rgb.A = GetRValue(Color);
	rgb.B = GetGValue(Color);
	rgb.C = GetBValue(Color);
	return rgb;
}

struct cABC RGB2HSV(struct cABC rgb)
{
	double var_R = rgb.A / 255.0f;
	double var_G = rgb.B / 255.0f;
	double var_B = rgb.C / 255.0f;

	double var_Min = Min(rgb) / 255.0f;
	double var_Max = Max(rgb) / 255.0f;
	double del_Max = var_Max - var_Min;

	cABC hsv;
	hsv.C = (BYTE)(var_Max * 255);

	if (del_Max == 0)
	{
		hsv.A = 0;
		hsv.B = 0;
	}
	else
	{
		hsv.B = (BYTE)((del_Max / var_Max) * 255);
		double del_R = ( ( ( var_Max - var_R ) / 6 ) + ( del_Max / 2 ) ) / del_Max;
		double del_G = ( ( ( var_Max - var_G ) / 6 ) + ( del_Max / 2 ) ) / del_Max;
		double del_B = ( ( ( var_Max - var_B ) / 6 ) + ( del_Max / 2 ) ) / del_Max;
		double H;
		if (var_R == var_Max) H = del_B - del_G;
		else if (var_G == var_Max) H = ( 1 / 3.0f ) + del_R - del_B;
		else if (var_B == var_Max) H = ( 2 / 3.0f ) + del_G - del_R;
		if (H < 0) H += 1;
		if (H > 1) H -= 1;
		hsv.A = (BYTE)(H * 255);
	}
	return hsv;
}

struct cABC HSV2RGB(struct cABC hsv)
{
	struct cABC rgb;
	if (hsv.B == 0)
		rgb.A = rgb.B = rgb.C = hsv.C;
	else
	{
		double var_h = hsv.A / 255.0f * 6;
		if (var_h == 6) var_h = 0;
		int var_i = int( var_h );
		double var_1 = (hsv.C / 255.0f) * ( 1 - (hsv.B / 255.0f) );
		double var_2 = (hsv.C / 255.0f) * ( 1 - (hsv.B / 255.0f) * ( var_h - var_i ) );
		double var_3 = (hsv.C / 255.0f) * ( 1 - (hsv.B / 255.0f) * ( 1 - ( var_h - var_i ) ) );

		double var_r, var_g, var_b;

		if		( var_i == 0 )	{ var_r = (hsv.C / 255.0f)     ; var_g = var_3 ; var_b = var_1; }
		else if ( var_i == 1 )	{ var_r = var_2 ; var_g = (hsv.C / 255.0f)     ; var_b = var_1; }
		else if ( var_i == 2 )	{ var_r = var_1 ; var_g = (hsv.C / 255.0f)     ; var_b = var_3; }
		else if ( var_i == 3 )	{ var_r = var_1 ; var_g = var_2 ; var_b = (hsv.C / 255.0f);     }
		else if ( var_i == 4 )	{ var_r = var_3 ; var_g = var_1 ; var_b = (hsv.C / 255.0f);     }
		else					{ var_r = (hsv.C / 255.0f)     ; var_g = var_1 ; var_b = var_2; };

		rgb.A = (BYTE)(var_r * 255);
		rgb.B = (BYTE)(var_g * 255);
		rgb.C = (BYTE)(var_b * 255);
	}
	return rgb;
}

unsigned long RGB2RGB(struct cABC abc)
{
	return RGB(abc.A, abc.B, abc.C);
}

unsigned int DiffSqr(BYTE A, BYTE B)
{
	BYTE x = A - B;
	BYTE y = min(256 - x, x);
	return y * y;
}

double Gauss(double mean, double sigma)
{
	double x, y, r2;
	const int max = 100000;
	do
	{
		x = (2.0 * (rand() % max)) / max - 1.0;
		y = (2.0 * (rand() % max)) / max - 1.0;
		r2 = x * x + y * y;
	} while ((r2 >= 1.0) || (r2 == 0));
	return (mean + sigma * y * sqrt(-2.0 * log(r2) / r2));
} 

void RealignColorMap(	unsigned long *cmColorMap, int nColors,
						unsigned char hue, double sat, unsigned int bandwidth)
{
	static const double C2D = 2 * M_PI / 255.0f;
	double tempy = 0, tempx = 0;
	for (int i = 0; i < nColors; i++)
	{
		tempy += sin(C2D * RGB2HSV(RGB2RGB(cmStdColorMap[i])).A);
		tempx += cos(C2D * RGB2HSV(RGB2RGB(cmStdColorMap[i])).A);
	}
	BYTE mean = ((BYTE)((atan2(
		min(max(tempy / nColors, -1), 1),
		min(max(tempx / nColors, -1), 1))) / C2D + 0.5));
	unsigned int derriv = 0;
	for (int i = 0; i < nColors; i++)
		derriv += DiffSqr(mean, RGB2HSV(RGB2RGB(cmStdColorMap[i])).A);
	derriv /= nColors;

	double rescale = 1;
	if (bandwidth > 0)
		rescale = bandwidth * bandwidth / (float)derriv;
	for (int i = 0; i < nColors; i++)
	{
		struct cABC hsv = RGB2HSV(RGB2RGB(cmStdColorMap[i]));
		hsv.A += max(128 - mean, mean - 128);
		hsv.A = (BYTE)((hsv.A - 128.0f)* rescale + 128.0f);
		hsv.A += max(hue - 128, hue - 128);
		hsv.B = (BYTE)(hsv.B * sat);
		cmColorMap[i] = RGB2RGB(HSV2RGB(hsv));
	}
}

void MakeColorMap(	unsigned long *cmColorMap, int nColors,
					unsigned char hue, unsigned int hsigma, unsigned int vssigma)
{
	for (int i = 0; i < nColors; i++)
	{
		struct cABC hsv;
		hsv.A = (BYTE)(Gauss(hue, hsigma));
		hsv.B = (BYTE)(Gauss(256, vssigma));
		hsv.C = (BYTE)(Gauss(256, vssigma));
		cmColorMap[i] = RGB2RGB(HSV2RGB(hsv));
	}
	for (int i = 0; i < nColors; i++)
		for (int j = i + 1; j < nColors; j++)
			if (RGB2HSV(RGB2RGB(cmColorMap[i])).C > RGB2HSV(RGB2RGB(cmColorMap[j])).C)
			{
				unsigned long temp = cmColorMap[i];
				cmColorMap[i] = cmColorMap[j];
				cmColorMap[j] = temp;
			}
}

BYTE Variation(BYTE base)
{
	signed char var = (rand() % 10 - 10);
	return (base + var);
}

void GetColorMap(DWORD idTheme, unsigned long *&cmColorMap, int &nColors)
{
	nColors = sizeof(cmStdColorMap) / sizeof(cmStdColorMap[0]);
	cmColorMap = new unsigned long[nColors];
	int value = Gauss(64, 16);
	switch (idTheme)
	{
	default:
		for (int i = 0; i < nColors; i++)
			cmColorMap[i] = cmStdColorMap[i];
		break;
	case IDS_THEME_RANDOM:
		MakeColorMap(cmColorMap, nColors, rand(), 20, 128);
//		RealignColorMap(cmColorMap, nColors, Variation(rand()), 1, 10);
		break;
	case IDS_THEME_RED:
		MakeColorMap(cmColorMap, nColors, 0, 20, 128);
//		RealignColorMap(cmColorMap, nColors, Variation(0), 1, 10);
		break;
	case IDS_THEME_BLUE:
		MakeColorMap(cmColorMap, nColors, 170, 20, 128);
//		RealignColorMap(cmColorMap, nColors, Variation(171), 2, 10);
		break;
	case IDS_THEME_GREEN:
		MakeColorMap(cmColorMap, nColors, 85, 20, 128);
//		RealignColorMap(cmColorMap, nColors, Variation(85), 2, 10);
		break;
	case IDS_THEME_GRAY:
		RealignColorMap(cmColorMap, nColors, Variation(rand()), 0.1, 10);
		break;
	case IDS_THEME_YELLOW:
		MakeColorMap(cmColorMap, nColors, 45, 20, 64);
//		RealignColorMap(cmColorMap, nColors, Variation(42), 2, 10);
		break;
	case IDS_THEME_CHAOS:
		for (int i = 0; i < nColors; i++)
		{
			struct cABC hsv;
			hsv.A = (BYTE)(rand());
			hsv.B = (BYTE)(Gauss(256, 128));
			hsv.C = (BYTE)(Gauss(256, 128));
			cmColorMap[i] = RGB2RGB(HSV2RGB(hsv));
		}
		break;
	case IDS_THEME_LIGHT:
		for (int i = 0; i < nColors; i++)
		{
			struct cABC hsv = RGB2HSV(RGB2RGB(cmStdColorMap[i]));
			hsv.C = (BYTE)((hsv.C + value < 255) ? hsv.C + value : 255);
			cmColorMap[i] = RGB2RGB(HSV2RGB(hsv));
		}
		break;
	case IDS_THEME_DARK:
		for (int i = 0; i < nColors; i++)
		{
			struct cABC hsv = RGB2HSV(RGB2RGB(cmStdColorMap[i]));
			hsv.C = (BYTE)((hsv.C - value > 0) ? hsv.C - value : 0);
			cmColorMap[i] = RGB2RGB(HSV2RGB(hsv));
		}
		break;
	}
}
