/*
 * Configuration functions
 * $Id: Config.cpp 417 2008-01-23 22:30:38Z fred $
 *
 * Copyright (c) 2006 Fred-Markus Stober <mail@fredstober.de>
 *
 * Permission is hereby granted, free of charge, to any person
 * obtaining a copy of this software and associated documentation
 * files (the "Software"), to deal in the Software without
 * restriction, including without limitation the rights to use,
 * copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following
 * conditions:

 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.

 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
 * OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
 * HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
 * WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
 * OTHER DEALINGS IN THE SOFTWARE.
 *
 */

#define WIN32_LEAN_AND_MEAN
#define _USE_MATH_DEFINES

#include <windows.h>
#include <strsafe.h>
#include <commctrl.h>
#include <commdlg.h>
#include <stdlib.h>
#include "Resource.h"
#include "Message.h"
#include "Config.h"

extern struct Config *pConfig;

static COLORREF crCustom[16];

static const char *szMyAppName = "Substrate";
static const char *szSpeed = "Speed";
static const char *szInitial = "InitialCracks";
static const char *szMax = "MaxCracks";
static const char *szGrains = "Grains";
static const char *szCircles = "Circles";
static const char *szCirclesDec = "DecreaseCircles";
static const char *szFore = "Foreground";
static const char *szBack = "Background";
static const char *szWireframe = "Wireframe";
static const char *szRandom = "Random";
static const char *szMaxSteps = "MaxSteps";
static const char *szTheme = "Theme";
static const char *szConfig = "Substrate.ini";

unsigned long Themelist[] =
{
	IDS_THEME_STD, IDS_THEME_RANDOM, IDS_THEME_RED, IDS_THEME_GREEN,
	IDS_THEME_BLUE, IDS_THEME_YELLOW, IDS_THEME_GRAY, IDS_THEME_CHAOS,
	IDS_THEME_LIGHT, IDS_THEME_DARK
};

void WriteConfigString(unsigned int value, const char *key)
{
	char szTemp[80];
	HRESULT hr = StringCchPrintf(szTemp, 20, "%ld", value);
	if (SUCCEEDED(hr))
		WritePrivateProfileString(szMyAppName, key, szTemp, szConfig);
}

int MinMax(int x)
{
	if (x < 0) return 0;
	if (x > 255) return 255;
	return x;
}

struct Config *InitConfigStruct()
{
	struct Config *pConfig = new Config();
	pConfig->steps = 0;
	if (GetPrivateProfileInt(szMyAppName, szRandom, 0, szConfig))
	{
		pConfig->wireframe = (rand() % 2) ? true : false;
		pConfig->circle_dec = (rand() % 2) ? true : false;
		int rnd = rand() % 128;
		pConfig->fgcolor = RGB(
			MinMax(rnd + (rand() % 32)),
			MinMax(rnd + (rand() % 32)),
			MinMax(rnd + (rand() % 32)));
		pConfig->fgcolor = RGB(0, 0, 0);
		rnd = rand() % 128 + 128;
		pConfig->bgcolor = RGB(
			MinMax(rnd - (rand() % 32)),
			MinMax(rnd - (rand() % 32)),
			MinMax(rnd - (rand() % 32)));
		pConfig->bgcolor = RGB(255, 255, 255);
		pConfig->grains = rand() % 128;
		pConfig->circle_percent = rand() % 100;
		pConfig->max_num = rand() % 150 + 5;
		pConfig->speed = rand() % 20 + 5;
		pConfig->initial_cracks = rand() % 10 + 2;
		pConfig->max_steps = rand() % 10000 + 2000;
		pConfig->theme = IDS_THEME_RANDOM;
	}
	else
	{
		pConfig->wireframe = (GetPrivateProfileInt(szMyAppName, szWireframe, 0, szConfig) == 0 ? false : true);
		pConfig->circle_dec = (GetPrivateProfileInt(szMyAppName, szCirclesDec, 1, szConfig) == 0 ? false : true);
		pConfig->fgcolor = GetPrivateProfileInt(szMyAppName, szFore, RGB(0,0,0), szConfig);
		pConfig->bgcolor = GetPrivateProfileInt(szMyAppName, szBack, RGB(255,255,255), szConfig);
		pConfig->grains = GetPrivateProfileInt(szMyAppName, szGrains, 64, szConfig);
		pConfig->circle_percent = GetPrivateProfileInt(szMyAppName, szCircles, 5, szConfig);
		pConfig->max_num = GetPrivateProfileInt(szMyAppName, szMax, 50, szConfig);
		pConfig->speed = GetPrivateProfileInt(szMyAppName, szSpeed, USER_TIMER_MINIMUM, szConfig);
		pConfig->initial_cracks = GetPrivateProfileInt(szMyAppName, szInitial, 3, szConfig);
		pConfig->max_steps = GetPrivateProfileInt(szMyAppName, szMaxSteps, 8000, szConfig);
		pConfig->theme = GetPrivateProfileInt(szMyAppName, szTheme, IDS_THEME_STD, szConfig);
	}
	return pConfig;
}

LRESULT SendTextMessage(HWND hWnd, UINT wmMessage, DWORD idText)
{
	TCHAR buffer[1024];
	LRESULT ret = FormatMessage(
		FORMAT_MESSAGE_FROM_HMODULE | FORMAT_MESSAGE_FROM_SYSTEM,
		NULL, idText, GetThreadLocale(), (LPTSTR)&buffer, 255, NULL);
	if (ret == 0)
		if (GetLastError() == ERROR_RESOURCE_LANG_NOT_FOUND)
			ret = FormatMessage(
				FORMAT_MESSAGE_FROM_HMODULE | FORMAT_MESSAGE_FROM_SYSTEM,
				NULL, idText, NULL, (LPTSTR)buffer, 255, NULL);
	ret = SendMessage(hWnd, wmMessage, NULL, (LPARAM)&buffer);
	return ret;
}

void AddThemes(HWND hWnd)
{
	for (int i = 0; i < sizeof(Themelist) / sizeof(unsigned long); i++)
	{
		LRESULT pos = SendTextMessage(hWnd, CB_ADDSTRING, Themelist[i]);
		SendMessage(hWnd, CB_SETITEMDATA , pos, (LPARAM)Themelist[i]);
	}
}

void SelectTheme(HWND hWnd, DWORD idTheme)
{
	LRESULT lPos = 0;
	for (unsigned int i = 0; i < sizeof(Themelist) / sizeof(unsigned long); i++)
		if (SendMessage(hWnd, CB_GETITEMDATA, i, 0) == idTheme) lPos = i;
	SendMessage(hWnd, CB_SETCURSEL, lPos, NULL);
}

void InitCheckBox(HWND hWnd, DWORD cControl, DWORD idText, bool bState)
{
	SendTextMessage(GetDlgItem(hWnd, cControl), WM_SETTEXT, idText);
	if (bState)
		SendMessage(GetDlgItem(hWnd, cControl), BM_SETCHECK, BST_CHECKED, NULL);
	else
		SendMessage(GetDlgItem(hWnd, cControl), BM_SETCHECK, BST_UNCHECKED, NULL);
}

void InitSlider(HWND hWnd, DWORD cControl, DWORD cLabel, DWORD idText, DWORD dwMin, DWORD dwMax, DWORD dwPos)
{
	SendTextMessage(GetDlgItem(hWnd, cLabel), WM_SETTEXT, idText);
	SendMessage(GetDlgItem(hWnd, cControl),	TBM_SETRANGE, false, MAKELONG(dwMin, dwMax));
	SendMessage(GetDlgItem(hWnd, cControl),	TBM_SETPOS, true, dwPos);
}

BOOL WINAPI ConfigureDialog(HWND hWnd, UINT idMessage,
							WPARAM wParam, LPARAM lParam)
{
	switch(idMessage)
	{
	case WM_INITDIALOG:
		if (pConfig == NULL)
			pConfig = InitConfigStruct();
		InitSlider(hWnd, IDC_SPEED,    IDL_SPEED,    IDS_SPEED,      1,    50, 51 - pConfig->speed);
		InitSlider(hWnd, IDC_MAXSTEPS, IDL_MAXSTEPS, IDS_MAXSTEPS, 500, 15000, pConfig->max_steps);
		InitSlider(hWnd, IDC_MAX,      IDL_MAX,      IDS_MAX,        5,   200, pConfig->max_num);
		InitSlider(hWnd, IDC_INITIAL,  IDL_INITIAL,  IDS_INITIAL,    1,    20, pConfig->initial_cracks);
		InitSlider(hWnd, IDC_CIRCLES,  IDL_CIRCLES,  IDS_CIRCLES,    0,   100, pConfig->circle_percent);
		InitSlider(hWnd, IDC_GRAIN,    IDL_GRAIN,    IDS_GRAIN,      0,   128, pConfig->grains);
		InitCheckBox(hWnd, IDC_CIRCLEDEC, IDS_CIRCLEDEC, pConfig->circle_dec);
		InitCheckBox(hWnd, IDC_WIREFRAME, IDS_WIREFRAME, pConfig->wireframe);
		InitCheckBox(hWnd, IDC_RANDOM, IDS_RANDOM, (GetPrivateProfileInt(szMyAppName, szRandom, 0, szConfig) == 0 ? false : true));
		SendTextMessage(GetDlgItem(hWnd, IDL_FORE), WM_SETTEXT, IDS_FORE);
		SendTextMessage(GetDlgItem(hWnd, IDL_BACK), WM_SETTEXT, IDS_BACK);
		SendTextMessage(GetDlgItem(hWnd, IDL_THEME), WM_SETTEXT, IDS_THEME);
		AddThemes(GetDlgItem(hWnd, IDC_THEME));
		SelectTheme(GetDlgItem(hWnd, IDC_THEME), pConfig->theme);
		return true;
	case WM_COMMAND:
		switch (LOWORD(wParam))
		{
		case IDC_OK:
			pConfig->speed = (unsigned int)(51 - SendMessage(GetDlgItem(hWnd, IDC_SPEED), TBM_GETPOS, NULL, NULL));
			WriteConfigString(pConfig->speed, szSpeed);
			pConfig->grains = (unsigned int)SendMessage(GetDlgItem(hWnd, IDC_GRAIN), TBM_GETPOS, NULL, NULL);
			WriteConfigString(pConfig->grains, szGrains);
			pConfig->circle_percent = (unsigned int)SendMessage(GetDlgItem(hWnd, IDC_CIRCLES), TBM_GETPOS, NULL, NULL);
			WriteConfigString(pConfig->circle_percent, szCircles);
			pConfig->max_num = (unsigned int)SendMessage(GetDlgItem(hWnd, IDC_MAX), TBM_GETPOS, NULL, NULL);
			WriteConfigString(pConfig->max_num, szMax);
			pConfig->max_steps = (unsigned int)SendMessage(GetDlgItem(hWnd, IDC_MAXSTEPS), TBM_GETPOS, NULL, NULL);
			WriteConfigString(pConfig->max_steps, szMaxSteps);
			pConfig->initial_cracks = (unsigned int)SendMessage(GetDlgItem(hWnd, IDC_INITIAL), TBM_GETPOS, NULL, NULL);
			WriteConfigString(pConfig->initial_cracks, szInitial);
			pConfig->wireframe = (SendMessage(GetDlgItem(hWnd, IDC_WIREFRAME), BM_GETCHECK, NULL, NULL) == BST_CHECKED);
			WriteConfigString(pConfig->wireframe, szWireframe);
			pConfig->circle_dec = (SendMessage(GetDlgItem(hWnd, IDC_WIREFRAME), BM_GETCHECK, NULL, NULL) == BST_CHECKED);
			WriteConfigString(pConfig->circle_dec, szCirclesDec);
			pConfig->theme = (int)SendMessage(GetDlgItem(hWnd, IDC_THEME), CB_GETITEMDATA,
				SendMessage(GetDlgItem(hWnd, IDC_THEME), CB_GETCURSEL, NULL, NULL), NULL);
			WriteConfigString(pConfig->theme, szTheme);

			WriteConfigString((SendMessage(GetDlgItem(hWnd, IDC_RANDOM), BM_GETCHECK, NULL, NULL) == BST_CHECKED), szRandom);

			WriteConfigString(pConfig->fgcolor, szFore);
			WriteConfigString(pConfig->bgcolor, szBack);

		case IDC_CANCEL:
			EndDialog(hWnd, LOWORD(wParam) == IDOK);
			return true;
		case IDC_BACK:
		case IDC_FORE:
			{
				CHOOSECOLOR cc;
				ZeroMemory(&cc, sizeof(cc));
				cc.lStructSize = sizeof(cc);
				cc.hwndOwner = hWnd;
				cc.lpCustColors = (LPDWORD)crCustom;
				cc.Flags = CC_FULLOPEN | CC_RGBINIT | CC_SOLIDCOLOR;
				if (LOWORD(wParam) == IDC_FORE)
					cc.rgbResult = pConfig->fgcolor;
				else
					cc.rgbResult = pConfig->bgcolor;
				if (ChooseColor(&cc) != NULL)
				{
					if (LOWORD(wParam) == IDC_FORE)
						pConfig->fgcolor = cc.rgbResult;
					else
						pConfig->bgcolor = cc.rgbResult;
					InvalidateRect(GetDlgItem(hWnd, LOWORD(wParam)), NULL, true);
				}
			}
		}
		break;
	case WM_DRAWITEM:
		{
			HBRUSH hBrush;
			LPDRAWITEMSTRUCT lpDIS;
			lpDIS = (LPDRAWITEMSTRUCT)lParam;
			switch (lpDIS->CtlID)
			{
			case IDC_FORE:
				hBrush = CreateSolidBrush(pConfig->fgcolor);
				break;
			case IDC_BACK:
				hBrush = CreateSolidBrush(pConfig->bgcolor);
				break;
			default:
				return false;
			}
			FillRect(lpDIS->hDC, &lpDIS->rcItem, hBrush);
			DeleteObject(hBrush);
			return true;
		}
	case WM_CLOSE:
		EndDialog(hWnd, false);
		return true;
	}
	return false;
}
