/*
 * DirectDraw utility functions
 * $Id: DDrawUtils.cpp 383 2007-05-07 13:25:27Z fred $
 *
 * Copyright (c) 2006 Fred-Markus Stober <mail@fredstober.de>
 *
 * Permission is hereby granted, free of charge, to any person
 * obtaining a copy of this software and associated documentation
 * files (the "Software"), to deal in the Software without
 * restriction, including without limitation the rights to use,
 * copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following
 * conditions:

 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.

 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
 * OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
 * HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
 * WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
 * OTHER DEALINGS IN THE SOFTWARE.
 *
 */

#include "Render.h"
#include "DDrawUtils.h"
#include <ddraw.h>

static LPDIRECTDRAW7 lpIDD7 = NULL;
static LPDIRECTDRAWSURFACE7 lpDDS = NULL;

static WORD RLT[256];
static WORD GLT[256];
static WORD BLT[256];
static int Pitch, PixelLen, Height, Width;
unsigned long *pOffscreen = NULL;

#define Pixel_Screen(x, y) (*(LPWORD)((DWORD)pSurface + y * Pitch + x * PixelLen))
#define Pixel_Screen_888(x, y) (*(LPDWORD)((DWORD)pSurface + y * Pitch + x * PixelLen))
#define Pixel_Offcreen(x, y) (pOffscreen[(y) * Width + (x)])

void DrawPixelColor_xxx(const WORD x, const WORD y, const DWORD Color);
void DrawPixelColor_888(const WORD x, const WORD y, const DWORD Color);
DrawPixelColorProc DrawPixelColor = DrawPixelColor_xxx;

void DrawPixelColorAlpha_xxx(const WORD x, const WORD y, const DWORD Color, double alpha);
void DrawPixelColorAlpha_888(const WORD x, const WORD y, const DWORD Color, double alpha);
DrawPixelColorAlphaProc DrawPixelColorAlpha = DrawPixelColorAlpha_xxx;

void DrawFadeOut_xxx(const DWORD Color, bool slow);
void DrawFadeOut_888(const DWORD Color, bool slow);
DrawFadeOutProc DrawFadeOut = DrawFadeOut_xxx;

void GetBitMaskInfo(DWORD Mask, CHAR &Pos, CHAR &Num)
{
	Pos = 0; Num = 8;
	while ((Mask & 1) == 0)
	{
		Mask >>= 1; Pos++;
	}
	while ((Mask & 1) == 1)
	{
		Mask >>= 1; Num--;
	}
}

DDPIXELFORMAT ddpf;

void DDInit(HWND hWnd)
{
	DDSURFACEDESC2 ddsd;
	if (FAILED(DirectDrawCreateEx(NULL, (LPVOID*)&lpIDD7, IID_IDirectDraw7, NULL)))
		exit(0);
	if (FAILED(lpIDD7->SetCooperativeLevel(hWnd, DDSCL_EXCLUSIVE | DDSCL_FULLSCREEN)))
		exit(0);
	ZeroMemory(&ddsd, sizeof(ddsd));
	ddsd.dwSize = sizeof(ddsd);
	ddsd.ddsCaps.dwCaps = DDSCAPS_PRIMARYSURFACE | DDSCAPS_SYSTEMMEMORY;
	ddsd.dwFlags = DDSD_CAPS;
	if (FAILED(lpIDD7->CreateSurface(&ddsd, &lpDDS, NULL)))
		exit(0);

	ZeroMemory(&ddpf, sizeof(ddpf));
	ddpf.dwSize = sizeof(ddpf);
	lpDDS->GetPixelFormat(&ddpf);

	switch (ddpf.dwRGBBitCount)
	{
	case 15:
	case 16:
		DrawFadeOut = DrawFadeOut_xxx;
		DrawPixelColor = DrawPixelColor_xxx;
		DrawPixelColorAlpha = DrawPixelColorAlpha_xxx;
		break;
	case 32:
		DrawFadeOut = DrawFadeOut_888;
		DrawPixelColor = DrawPixelColor_888;
		DrawPixelColorAlpha = DrawPixelColorAlpha_888;
		break;
	default:
		exit(0);
	}

	CHAR PosR, PosG, PosB, ShiftR, ShiftG, ShiftB;
	GetBitMaskInfo(ddpf.dwRBitMask, PosR, ShiftR);
	GetBitMaskInfo(ddpf.dwGBitMask, PosG, ShiftG);
	GetBitMaskInfo(ddpf.dwBBitMask, PosB, ShiftB);
	for (int c = 0; c < 256; c++)
	{
		RLT[c] = (c >> ShiftR) << PosR;
		GLT[c] = (c >> ShiftG) << PosG;
		BLT[c] = (c >> ShiftB) << PosB;
	}

	Width = GetSystemMetrics(SM_CXSCREEN);
	Height = GetSystemMetrics(SM_CYSCREEN);
	if (pOffscreen != NULL)
		delete [] pOffscreen;
	pOffscreen = new unsigned long[Width * Height];
	memset(pOffscreen, 0, sizeof(unsigned long) * Width * Height);
}

void DDExit()
{
	if (pOffscreen != NULL)
	{
		delete [] pOffscreen;
		pOffscreen = NULL;
	}
	if (lpIDD7 != NULL)
	{
		lpIDD7->Release();
		lpIDD7 = NULL;
	}
	if (lpDDS != NULL)
	{
		lpDDS->Release();
		lpDDS = NULL;
	}
}

LPBYTE pSurface;

void DDBeginPixelDraw()
{
	DDSURFACEDESC2 ddsd;
	ZeroMemory(&ddsd, sizeof(ddsd));
	ddsd.dwSize = sizeof(ddsd);
	HRESULT hr = lpDDS->Lock(0, &ddsd, DDLOCK_WAIT | DDLOCK_SURFACEMEMORYPTR, 0);
	if (hr != DD_OK)
		exit(1);
	Pitch = ddsd.lPitch;
	PixelLen = ddsd.ddpfPixelFormat.dwRGBBitCount >> 3;
	pSurface = (LPBYTE)ddsd.lpSurface;
}

void DDEndPixelDraw()
{
	pSurface = NULL;
	lpDDS->Unlock(NULL);
}

void DDScreenShot(HWND hWnd)
{
	HDC SurfDC = NULL;
	HDC OffscrDC = NULL;
	int width = GetSystemMetrics(SM_CXSCREEN);
	int height = GetSystemMetrics(SM_CYSCREEN);
	HBITMAP OffscrBmp = NULL;
	if (!FAILED(lpDDS->GetDC(&SurfDC)))
		if ((OffscrBmp = CreateCompatibleBitmap(SurfDC, width, height)) != NULL)
			if ((OffscrDC = CreateCompatibleDC(SurfDC)) != NULL)
			{
				HBITMAP OldBmp = (HBITMAP)SelectObject(OffscrDC, OffscrBmp);
				BitBlt(OffscrDC, 0, 0, width, height, SurfDC, 0, 0, SRCCOPY);
				lpDDS->ReleaseDC(SurfDC);
				SurfDC = NULL;
				SelectObject(OffscrDC, OldBmp);
				OpenClipboard(hWnd);
				SetClipboardData(CF_BITMAP, OffscrBmp);
				CloseClipboard();
			}
	if (SurfDC) lpDDS->ReleaseDC(SurfDC);
	if (OffscrDC) DeleteDC(OffscrDC);
	if (OffscrBmp) DeleteObject(OffscrBmp);
}

#define RGBtoXXX(R, G, B) (RLT[R] | GLT[G] | BLT[B])

void DrawPixelColor_xxx(const WORD x, const WORD y, const DWORD Color)
{
	Pixel_Offcreen(x, y) = Color;
	Pixel_Screen(x, y) = RGBtoXXX(GetRValue(Color), GetGValue(Color), GetBValue(Color));
}

void DrawPixelColor_888(const WORD x, const WORD y, const DWORD Color)
{
	Pixel_Offcreen(x, y) = Color;
	Pixel_Screen_888(x, y) = RGB(GetBValue(Color), GetGValue(Color), GetRValue(Color));;
}

inline void DrawPixelRGB_xxx(const WORD x, const WORD y, const BYTE r, const BYTE g, const BYTE b)
{
	Pixel_Offcreen(x, y) = RGB(r, g, b);
	Pixel_Screen(x, y) = RGBtoXXX(r, g, b);
}

inline void DrawPixelRGB_888(const WORD x, const WORD y, const BYTE r, const BYTE g, const BYTE b)
{
	Pixel_Offcreen(x, y) = RGB(r, g, b);
	Pixel_Screen_888(x, y) = RGB(b, g, r);
}

void DrawPixelColorAlpha_xxx(const WORD x, const WORD y, const DWORD Color, const double alpha)
{
	if ((x >= 0) && (x < Width) && (y >= 0) && (y < Height) && (alpha <= 1.0))
	{
		unsigned char *mc = (unsigned char *)&Color;
		unsigned char *c = (unsigned char *)&Pixel_Offcreen(x, y);
		unsigned char nr = (char)(*c + (*mc++ - *c++) * alpha);
		unsigned char ng = (char)(*c + (*mc++ - *c++) * alpha);
		unsigned char nb = (char)(*c + (*mc - *c) * alpha);
		DrawPixelRGB_xxx(x, y, nr, ng, nb);
	}
}

void DrawPixelColorAlpha_888(const WORD x, const WORD y, const DWORD Color, const double alpha)
{
	if ((x >= 0) && (x < Width) && (y >= 0) && (y < Height) && (alpha <= 1.0))
	{
		unsigned char *mc = (unsigned char *)&Color;
		unsigned char *c = (unsigned char *)&Pixel_Offcreen(x, y);
		unsigned char nr = (char)(*c + (*mc++ - *c++) * alpha);
		unsigned char ng = (char)(*c + (*mc++ - *c++) * alpha);
		unsigned char nb = (char)(*c + (*mc - *c) * alpha);
		DrawPixelRGB_888(x, y, nr, ng, nb);
	}
}

#define ComputeNewPixel(PTR, SHIFT, COL) (*PTR = (unsigned char)(*PTR - (*PTR >> SHIFT) + COL))

void DrawFadeOut_xxx(const DWORD Color, bool slow)
{
	BeginPixelDraw();
	WORD x = 0, y = 0;
	unsigned char *current = (unsigned char *)&Pixel_Offcreen(0, 0);
	unsigned long *finish = &Pixel_Offcreen(Width - 1, Height - 1);
	unsigned char r, g, b;
	if (slow)
	{
		r = GetRValue(Color) >> 3; g = GetGValue(Color) >> 3; b = GetBValue(Color) >> 3;
		while ((unsigned long *)current < finish)
		{
			unsigned char nr = ComputeNewPixel(current, 3, r); current++;
			unsigned char ng = ComputeNewPixel(current, 3, g); current++;
			unsigned char nb = ComputeNewPixel(current, 3, b); current++;
			Pixel_Screen(x, y) = RGBtoXXX(nr, ng, nb);
			current++; if (++x == Width) { x = 0; y++; }
		}
	}
	else
	{
		r = GetRValue(Color) >> 2; g = GetGValue(Color) >> 2; b = GetBValue(Color) >> 2;
		while ((unsigned long *)current < finish)
		{
			unsigned char nr = ComputeNewPixel(current, 2, r); current++;
			unsigned char ng = ComputeNewPixel(current, 2, g); current++;
			unsigned char nb = ComputeNewPixel(current, 2, b); current++;
			Pixel_Screen(x, y) = RGBtoXXX(nr, ng, nb);
			current++; if (++x == Width) { x = 0; y++; }
		}
	}
	EndPixelDraw();
}

void DrawFadeOut_888(const DWORD Color, bool slow)
{
	BeginPixelDraw();
	WORD x = 0, y = 0;
	unsigned char *current = (unsigned char *)&Pixel_Offcreen(0, 0);
	unsigned long *finish = &Pixel_Offcreen(Width - 1, Height - 1);
	unsigned char r, g, b;
	if (slow)
	{
		r = GetRValue(Color) >> 3; g = GetGValue(Color) >> 3; b = GetBValue(Color) >> 3;
		while ((unsigned long *)current < finish)
		{
			unsigned char nr = ComputeNewPixel(current, 3, r); current++;
			unsigned char ng = ComputeNewPixel(current, 3, g); current++;
			unsigned char nb = ComputeNewPixel(current, 3, b); current++;
			Pixel_Screen_888(x, y) = RGB(nb, ng, nr);
			current++; if (++x == Width) { x = 0; y++; }
		}
	}
	else
	{
		r = GetRValue(Color) >> 2; g = GetGValue(Color) >> 2; b = GetBValue(Color) >> 2;
		while ((unsigned long *)current < finish)
		{
			unsigned char nr = ComputeNewPixel(current, 2, r); current++;
			unsigned char ng = ComputeNewPixel(current, 2, g); current++;
			unsigned char nb = ComputeNewPixel(current, 2, b); current++;
			Pixel_Screen_888(x, y) = RGB(nb, ng, nr);
			current++; if (++x == Width) { x = 0; y++; }
		}
	}
	EndPixelDraw();
}

void DDDrawBackground(const DWORD Color)
{
	DDBLTFX ddbf;
	ZeroMemory(&ddbf, sizeof(ddbf));
	ddbf.dwSize = sizeof(ddbf);
	if (ddpf.dwRGBBitCount < 24)
		ddbf.dwFillColor = RGBtoXXX(GetRValue(Color), GetGValue(Color), GetBValue(Color));
	else
		ddbf.dwFillColor = Color;
	lpDDS->Blt(NULL, NULL, NULL, DDBLT_COLORFILL | DDBLT_WAIT, &ddbf);
	memset(pOffscreen, Color, sizeof(unsigned long) * Width * Height);
}

void DDDrawText(const char *text)
{
	HDC hDC;
	DDBLTFX ddbf;
	RECT rect = { 0, 0, 150, 20};
	ZeroMemory(&ddbf, sizeof(ddbf));
	ddbf.dwSize = sizeof(ddbf);
	ddbf.dwFillColor = RGBtoXXX(255, 255, 255);
	lpDDS->Blt(&rect, NULL, NULL, DDBLT_COLORFILL | DDBLT_WAIT, &ddbf);
	if (lpDDS->GetDC(&hDC) == DD_OK)
	{
		SetTextColor(hDC, RGB(0,0,0));
		TextOut(hDC, 0, 0, text, (int)strlen(text));
		lpDDS->ReleaseDC(hDC);
	};
}
