/*
 * Substrate implementation for Windows, Copyright (c) 2006 Fred-Markus Stober <mail@fredstober.de>
 * $Id: Substrate.cpp 417 2008-01-23 22:30:38Z fred $
 *
 * Directly ported code from xscreensaver Substrate code
 * http://www.jwz.org/xscreensaver/xscreensaver-5.02.tar.gz
 * which is directly ported code from complexification.net Substrate art
 * http://complexification.net/gallery/machines/substrate/applet_s/substrate_s.pde
 *
 * Substrate implementation for xscreensaver, Copyright (c) 2004 Mike Kershaw <dragorn@kismetwireless.net>
 * Substrate code, Copyright (c) 2004 Jared Tarbell <j.tarbell@complexification.net>
 * xscreensaver, Copyright (c) 1997, 1998, 2002 Jamie Zawinski <jwz@jwz.org>
 *
 * Permission to use, copy, modify, distribute, and sell this software and its
 * documentation for any purpose is hereby granted without fee, provided that
 * the above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation.  No representations are made about the suitability of this
 * software for any purpose.  It is provided "as is" without express or 
 * implied warranty.
 *
 * TODO: Multimon support, D3D rendering, trig lookup table
 *
 */

#define WIN32_LEAN_AND_MEAN
#define _USE_MATH_DEFINES

#include <windows.h>
#include <scrnsave.h>
#include <stdlib.h>
#include <math.h>
#include <time.h>
#include "Config.h"
#include "Render.h"
#include "ColorTable.h"
#include "resource.h"
#include "message.h"

#define SAFE_DELETE_ARRAY(X) { if (X != NULL) { delete [] X; (X) = NULL; } }
#define frand(MIN,MAX) ((((MAX - MIN) * rand()) / RAND_MAX) + MIN)

HBITMAP preview = NULL;
extern BOOL fChildPreview;
extern HINSTANCE hMainInstance;

static const double R_STEP = 0.42f;
static const double MAX_GRAINS = 1.0f;
static const double PI_180 = M_PI / 180.0f;

struct crack
{
	/* Synthesis of data from Crack:: and SandPainter:: */
	double x, y;
	double ys, xs; /* for curvature calculations */
	double t, t_inc;

	bool curved;

	unsigned long sandcolor;
	double sandp, sandg;

	double degrees_drawn;
};

unsigned int height;
unsigned int width;

struct Config *pConfig;

/* grid of cracks */
crack *cracks = NULL;
int num_cracks;
int max_cracks;
/* grid of actual crack placement */
int *cgrid = NULL;
/* Grains */
int num_grains;
/* Color lookup table */
unsigned long *colormap = NULL;
int num_colors;

/* Quick references to pixels in the offscreen map and in the crack grid */
#define ref_cgrid(x, y) (cgrid[(y) * width + (x)])

inline void start_crack(struct Config *config, crack *cr)
{
	/* synthesis of Crack::findStart() and crack::startCrack() */
	int px = 0;
	int py = 0;
	int found = 0;
	int timeout = 0;
	double a;

	/* shift until crack is found */
	while ((!found) && (timeout++ < 10000))
	{
		px = (int)(rand() % width);
		py = (int)(rand() % height);

		if (ref_cgrid(px, py) < 10000)
			found = 1;
	}

	if (!found)
	{
		/* We timed out.  Use our default values */
		px = (int)cr->x;
		py = (int)cr->y;
		ref_cgrid(px, py) = (int)cr->t;
	}

	/* start a crack */
	a = ref_cgrid(px, py);

	if ((rand() % 100) < 50)
		a -= 90 + frand(-2.0f, 2.1f);
	else
		a += 90 + frand(-2.0f, 2.1f);

	if ((rand() % 100) < pConfig->circle_percent)
	{
		double r; /* radius */
		double radian_inc;

		cr->curved = true;
		cr->degrees_drawn = 0;

		r = 10 + (rand() % ((width + height) / 2));

		if ((rand() % 100) < 50)
			r *= -1;

		/* arc length = r * theta => theta = arc length / r */
		radian_inc = R_STEP / r;
		cr->t_inc = radian_inc / PI_180;

		cr->ys = r * sin(radian_inc);
		cr->xs = r * (1 - cos(radian_inc));
		if ((pConfig->circle_dec) && (pConfig->circle_percent >= 5))
			pConfig->circle_percent -= 5;
	}
	else
		cr->curved = 0;

	/* Condensed from Crack::startCrack */
	cr->x = px + (0.61 * cos(a * PI_180));
	cr->y = py + (0.61 * sin(a * PI_180));
	cr->t = a;
	cr->sandcolor = colormap[rand() % num_colors];
}

inline void make_crack(struct Config *config)
{
	crack *cr;

	if (num_cracks < max_cracks)
	{
		/* make a new crack */
		cr = &(cracks[num_cracks]);
		/* assign colors */
		cr->sandp = 0;
		cr->sandg = frand(-0.01, 0.01);
		cr->sandcolor = colormap[rand() % num_colors];
		cr->curved = 0;
		cr->degrees_drawn = 0;

		/* We could use these values in the timeout case of start_crack */

		cr->x = rand() % width;
		cr->y = rand() % height;
		cr->t = rand() % 360;

		/* start it */
		start_crack(config, cr);

		num_cracks++;
	}
}

inline void region_color(struct Config *config, crack *cr)
{
	/* synthesis of Crack::regionColor() and SandPainter::render() */

	double rx = cr->x;
	double ry = cr->y;
	int openspace = 1;
	unsigned int cx, cy;
	int i;
	double drawx, drawy;

	while (openspace)
	{
		/* move perpendicular to crack */
		rx += (0.81 * sin(cr->t * PI_180));
		ry -= (0.81 * cos(cr->t * PI_180));

		cx = (int) rx;
		cy = (int) ry;

		if ((cx >= 0) && (cx < width) && (cy >= 0) && (cy < height))
		{
			/* safe to check */
			if (cgrid[cy * width + cx] <= 10000)
				openspace = 0;
		}
		else
			openspace = 0;
	}

	/* SandPainter stuff here */

	/* Modulate gain */
	cr->sandg += frand(-0.05, 0.05);

	if (cr->sandg < 0)
		cr->sandg = 0;

	if (cr->sandg > MAX_GRAINS)
		cr->sandg = MAX_GRAINS;

	/* Lay down grains of sand */
	double w = cr->sandg / (num_grains - 1);
	for (i = 0; i < num_grains; i++)
	{
		drawx = (cr->x + (rx - cr->x) * sin(cr->sandp + sin(i * w)));
		drawy = (cr->y + (ry - cr->y) * sin(cr->sandp + sin(i * w)));
		/* Draw sand bit */
		DrawPixelColorAlpha((WORD)drawx, (WORD)drawy, cr->sandcolor, (0.1f - i / (num_grains * 10.0f)));
	}
}

void CreateSubstrate(struct Config *config)
{
	num_cracks = 0;
	pConfig->steps = 0;
	max_cracks = pConfig->max_num;
	num_grains = pConfig->grains;
	DrawBackground(pConfig->bgcolor);

	/* get themed colormap */
	SAFE_DELETE_ARRAY(colormap);
	GetColorMap(pConfig->theme, colormap, num_colors);

	/* erase crack grid */
	SAFE_DELETE_ARRAY(cgrid);
	cgrid = new int[height * width];
	memset(cgrid, 10001, height * width * sizeof(int));

	/* erase crack array */
	SAFE_DELETE_ARRAY(cracks);
	cracks = new crack[max_cracks + 1];

	/* create new cracks */
	for (unsigned int tx = 0; tx < pConfig->initial_cracks; tx++)
		make_crack(config);
}

inline void movedrawcrack(struct Config *config, int cracknum)
{
	/* Basically Crack::move() */
	unsigned int cx, cy;
	crack *cr = &(cracks[cracknum]);

	/* continue cracking */
	if (!cr->curved)
	{
		cr->x += (R_STEP * cos(cr->t * PI_180));
		cr->y += (R_STEP * sin(cr->t * PI_180));
	}
	else
	{
		cr->x += cr->ys * cos(cr->t * PI_180) + cr->xs * sin(cr->t * PI_180);
		cr->y += cr->ys * sin(cr->t * PI_180) - cr->xs * cos(cr->t * PI_180);
		cr->t += cr->t_inc;
		cr->degrees_drawn += abs(cr->t_inc);
	}

	cx = (int) (cr->x + frand(-0.33, 0.33));
	cy = (int) (cr->y + frand(-0.33, 0.33));

	/* bounds check */
	if ((cx >= 0) && (cx < width) && (cy >= 0) && (cy < height))
	{
		/* draw sand painter if we're not wireframe */
		if (!pConfig->wireframe)
			region_color(config, cr);

		/* draw fgcolor crack */
		DrawPixelColor(cx, cy, pConfig->fgcolor);

		if (cr->curved && (cr->degrees_drawn > 360) )
		{
			/* completed the circle, stop cracking */
			start_crack(config, cr);		/* restart ourselves */
			make_crack(config);			/* generate a new crack */
		}
		/* safe to check */
		else
			if ((cgrid[cy * width + cx] > 10000) ||
				(abs(cgrid[cy * width + cx] - cr->t) < 5))
			{
				/* continue cracking */
				cgrid[cy * width + cx] = (int) cr->t;
			}
			else
				if (abs(cgrid[cy * width + cx] - cr->t) > 2)
				{
					/* crack encountered (not self), stop cracking */
					start_crack(config, cr); /* restart ourselves */
					make_crack(config); /* generate a new crack */
				}
	}
	else
	{
		/* out of bounds, stop cracking */
		/* need these in case of timeout in start_crack */
		cr->x = rand() % width;
		cr->y = rand() % height;
		cr->t = rand() % 360;
		start_crack(config, cr); /* restart ourselves */
		make_crack(config); /* generate a new crack */
	}
}

#ifdef _DEBUG
#include <stdio.h>

char fps[100];
LARGE_INTEGER time_freq, time_old, time_new, time_start;
#endif

LONG WINAPI ScreenSaverProc(HWND hWnd, UINT idMessage, WPARAM wParam, LPARAM lParam)
{
#ifdef _DEBUG
	if ((idMessage == WM_LBUTTONDOWN) ||
		(idMessage == WM_RBUTTONDOWN) ||
		(idMessage == WM_MBUTTONDOWN) ||
		(idMessage == WM_MOUSEMOVE) ||
		(idMessage == WM_SETCURSOR))
		return 0;
#endif
	switch(idMessage) 
	{ 
	case WM_CREATE:
		if (fChildPreview)
		{
			preview = LoadBitmap(hMainInstance, MAKEINTRESOURCE(IDB_PREVIEW));
			break;
		}
		InitRenderer(hWnd);
#ifdef _DEBUG
		srand(1);
		QueryPerformanceFrequency(&time_freq);
		QueryPerformanceCounter(&time_start);
#else
		srand((unsigned int)time(NULL));
#endif
		if (pConfig == NULL)
		{
			width = GetSystemMetrics(SM_CXSCREEN);
			height = GetSystemMetrics(SM_CYSCREEN);
			pConfig = InitConfigStruct();
		}
		CreateSubstrate(pConfig);
		SetTimer(hWnd, 0, pConfig->speed, 0);
		break;
	case WM_ERASEBKGND:
		return true;
	case WM_PAINT:
		if (fChildPreview)
		{
			RECT rc;
			PAINTSTRUCT ps;
			GetClientRect(hWnd, &rc);
			HDC hDC = BeginPaint(hWnd, &ps);
			HDC hMem = CreateCompatibleDC(hDC);
			SelectObject(hMem, preview);
			BitBlt(hDC, 0, 0, rc.right, rc.bottom, hMem, 0, 0, SRCCOPY);
			DeleteDC(hMem);
			EndPaint(hWnd, &ps);
		}
		break;
	case WM_KEYDOWN:
		if (wParam == VK_SNAPSHOT)
			ScreenShot(hWnd);
		break;
	case WM_TIMER:
#ifdef _DEBUG
		{
			if (pConfig->steps < 1000)
			{
				QueryPerformanceCounter(&time_new);
				sprintf(fps, "step %5i, %.2f fps",
					pConfig->steps, time_freq.QuadPart / (double)(time_new.QuadPart - time_old.QuadPart));
				DrawText(fps);
				time_old = time_new;
				if (pConfig->steps % 100 == 0)
					for (int t = IDS_THEME_STD; t <= IDS_THEME_DARK; t++)
					{
						GetColorMap(t, colormap, num_colors);
						for (int i = 0; i < num_colors; i++)
						{
							BeginPixelDraw();
							for	(int j = 0; j < 10*10; j++)
								DrawPixelColorAlpha((i % 10) * 10 + j % 10 + (t - IDS_THEME_STD) * 110, (i / 10) * 10 + j / 10 + 50, colormap[i], 1);
							EndPixelDraw();
						}
					}
				GetColorMap(pConfig->theme, colormap, num_colors);
			}
			else
			{
				if (pConfig->steps == 1000)
					ScreenShot(hWnd);
				sprintf(fps, "%.3f s", (double)(time_new.QuadPart - time_start.QuadPart) / time_freq.QuadPart);
				DrawText(fps);
			}
		}
#endif
		if (pConfig->steps == 1) DrawBackground(pConfig->bgcolor); // BUGFIX: DrawBackground again
		if ((pConfig->steps < pConfig->max_steps) || (pConfig->max_steps == 30000))
		{
			BeginPixelDraw();
			DrawPixelColor(width - 1, height - 1, RGB(255,255,255));
			for (int tempx = 0; tempx < num_cracks; tempx++)
				movedrawcrack(pConfig, tempx);
			if (pConfig->speed < USER_TIMER_MINIMUM)
				for (unsigned int times = 0; times < USER_TIMER_MINIMUM - pConfig->speed; times++)
					for (int tempx = 0; tempx < num_cracks; tempx++)
						movedrawcrack(pConfig, tempx);
			EndPixelDraw();
		}
		else
			if (pConfig->steps < pConfig->max_steps + 32)
				if (pConfig->steps < pConfig->max_steps + 20)
					DrawFadeOut(pConfig->bgcolor, true);
				else
					DrawFadeOut(pConfig->bgcolor, false);
			else
				CreateSubstrate(pConfig);
		pConfig->steps++;
		break;
	case WM_DESTROY: 
		KillTimer(hWnd, NULL);
		PostQuitMessage(0);
		ExitRenderer();
		return 0;
	}
	return DefScreenSaverProc(hWnd, idMessage, wParam, lParam);
}

BOOL WINAPI ScreenSaverConfigureDialog(HWND hWnd, UINT idMessage, WPARAM wParam, LPARAM lParam)
{
	return ConfigureDialog(hWnd, idMessage, wParam, lParam);
}

BOOL WINAPI RegisterDialogClasses(HANDLE hInst)
{
	return true;
}
